﻿unit StoreRegistry;

Interface

uses

  Winapi.Windows,
  Winapi.Messages,
  System.SysUtils,
  System.Classes,
  System.Win.Registry;

type TStoreRegistry = class

  const REGISTRY_ROOT_PATH = 'SOFTWARE\\VaxPhoneSDKv6';

  private
  m_hRootKey: TRegistry;

  public
  constructor Create(); reintroduce; overload;
  Destructor Destroy(); reintroduce; overload;

  private
  procedure Open();
  procedure Close();

  private
  function CreateKey(sKeyName: String): TRegistry;
  function OpenKey(sKeyName: String): TRegistry;

  public
  function SetTEXT(sKeyName: String; sValueName: String; sValueData: String): Boolean;
  function SetINT(sKeyName: String; sValueName: String; nValueData: Integer): Boolean;
  function SetBOOL(sKeyName: String; sValueName: String; bValueData: Boolean): Boolean;

  procedure GetTEXT(sKeyName: String; sValueName: String; var sValueData: String; sValueDefault: String);
  procedure GetINT(sKeyName: String; sValueName: String; var nValueData: Integer; nValueDefault: Integer);
  procedure GetBOOL(sKeyName: String; sValueName: String; var bValueData: Boolean; bValueDefault: Boolean);

  procedure DeleteSubKey(sKeyName: String);
  function DeleteValue(sKeyRootName: String; sValueName: String): Boolean;

  function EnumSubKeys(sKeyRootName: String; var objList: TStringList): Boolean;
  function EnumValueNames(sKeyRootName: String; var objList: TStringList): Boolean;

  function IsKeyExist(sKeyName: String): Boolean;

end;


implementation

  //////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////

  constructor TStoreRegistry.Create();
  begin
    m_hRootKey := nil;
    Open();
  end;

  Destructor TStoreRegistry.Destroy();
  begin
    Close();
  end;

  //////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////

  procedure TStoreRegistry.Open();
  begin

    if(m_hRootKey <> nil) then
      Exit();

    m_hRootKey := TRegistry.Create();
    m_hRootKey.RootKey := HKEY_CURRENT_USER;

    m_hRootKey.OpenKey(REGISTRY_ROOT_PATH, True);

  end;

  procedure TStoreRegistry.Close();
  begin
    if(m_hRootKey = nil) then
      Exit();

    m_hRootKey.CloseKey();
    m_hRootKey.Destroy();

    m_hRootKey := nil;
  end;

  //////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////

  function TStoreRegistry.CreateKey(sKeyName: String): TRegistry;
  var
    hKey: TRegistry;
  begin

    if(m_hRootKey = nil) then
      Exit(nil);

    hKey := TRegistry.Create();
    hKey.RootKey := m_hRootKey.CurrentKey;

    if(hKey.OpenKey(sKeyName, True) = False) then Exit(nil);

    Result := hKey;

  end;

  function TStoreRegistry.OpenKey(sKeyName: String): TRegistry;
  var
    hKey: TRegistry;
  begin

    if(m_hRootKey = nil) then
      Exit(nil);

    hKey := TRegistry.Create();
    hKey.RootKey := m_hRootKey.CurrentKey;

    if(hKey.OpenKey(sKeyName, False) = False) then Exit(nil);

    Result := hKey;

  end;

  //////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////

  function TStoreRegistry.SetTEXT(sKeyName: String; sValueName: String; sValueData: String): Boolean;
  var
    hKey: TRegistry;
  begin

    if Length(sValueName) = 0 then
    begin
      Exit(False);
    end;

    hKey := CreateKey(sKeyName);
    if(hKey = nil) Then Exit(False);

    hKey.WriteString(sValueName, sValueData);
    hKey.Destroy();

    Result := True;

  end;

  function TStoreRegistry.SetINT(sKeyName: String; sValueName: String; nValueData: Integer): Boolean;
  var
    hKey: TRegistry;
  begin

    if Length(sValueName) = 0 then
    begin
      Exit(False);
    end;

    hKey := CreateKey(sKeyName);
    if(hKey = nil) Then Exit(False);

    hKey.WriteInteger(sValueName, nValueData);
    hKey.Destroy();

    Result := True;

  end;

  function TStoreRegistry.SetBOOL(sKeyName: String; sValueName: String; bValueData: Boolean): Boolean;
  var
    hKey: TRegistry;
  begin

    if Length(sValueName) = 0 then
    begin
      Exit(False);
    end;

    hKey := CreateKey(sKeyName);
    if(hKey = nil) Then Exit(False);

    hKey.WriteBool(sValueName, bValueData);
    hKey.Destroy();

    Result := True;

  end;

  //////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////

  procedure TStoreRegistry.GetTEXT(sKeyName: String; sValueName: String; var sValueData: String; sValueDefault: String);
  var
    hKey: TRegistry;
  begin

    if Length(sValueName) = 0 then
    begin
      sValueData:= sValueDefault;
      Exit();
    end;

    hKey := OpenKey(sKeyName);
    if(hKey = nil) Then
    begin
      sValueData := sValueDefault;
      Exit();
    end;

    try
      sValueData := hKey.ReadString(sValueName);
    except
     on E:ERegistryException do
     begin
      sValueData := sValueDefault;
      hKey.Destroy();
      Exit();
     end;
    end;

    hKey.Destroy();

  end;

  procedure TStoreRegistry.GetINT(sKeyName: String; sValueName: String; var nValueData: Integer; nValueDefault: Integer);
  var
    hKey: TRegistry;
  begin

    if Length(sValueName) = 0 then
    begin
      nValueData := nValueDefault;
      Exit();
    end;

    hKey := OpenKey(sKeyName);
    if(hKey = nil) Then
    begin
      nValueData := nValueDefault;
      Exit();
    end;

    try
      nValueData := hKey.ReadInteger(sValueName);
    except
     on E:ERegistryException do
      nValueData := nValueDefault;
    end;

    hKey.Destroy();

  end;

  procedure TStoreRegistry.GetBOOL(sKeyName: String; sValueName: String; var bValueData: Boolean; bValueDefault: Boolean);
  var
    hKey: TRegistry;
  begin

    if Length(sValueName) = 0 then
    begin
      bValueData := bValueDefault;
      Exit();
    end;

    hKey := OpenKey(sKeyName);
    if(hKey = nil) Then
    begin
      bValueData := bValueDefault;
      Exit();
    end;

    try
      bValueData := hKey.ReadBool(sValueName);
    except
     on E:ERegistryException do
      bValueData := bValueDefault;
    end;

    hKey.Destroy();

  end;

  //////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////

  procedure TStoreRegistry.DeleteSubKey(sKeyName: String);
  var
    hKey: TRegistry;
  begin

    if(m_hRootKey = nil) then
      Exit();

    hKey := TRegistry.Create();
    hKey.RootKey := m_hRootKey.CurrentKey;

    hKey.DeleteKey(sKeyName);
    hKey.Destroy();

  end;

  function TStoreRegistry.DeleteValue(sKeyRootName: String; sValueName: String): Boolean;
  var
    hKey: TRegistry;
  begin

    if Length(sValueName) = 0 then
    begin
      Exit(False);
    end;

    hKey := OpenKey(sKeyRootName);
    if(hKey = nil) Then
    begin
      Exit(False);
    end;

    hKey.DeleteValue(sValueName);
    hKey.Destroy();

    Result := True;

  end;

  function TStoreRegistry.EnumSubKeys(sKeyRootName: String; var objList: TStringList): Boolean;
  var
    hKey: TRegistry;
  begin

    objList.Clear();

    hKey := OpenKey(sKeyRootName);
    if(hKey = nil) Then
    begin
      Exit(False);
    end;

    hKey.GetKeyNames(objList);
    hKey.Destroy();

    Result := True;

  end;

  function TStoreRegistry.EnumValueNames(sKeyRootName: String; var objList: TStringList): Boolean;
  var
    hKey: TRegistry;
  begin

    objList.Clear();

    hKey := OpenKey(sKeyRootName);
    if(hKey = nil) Then
    begin
      Exit(False);
    end;

    hKey.GetValueNames(objList);
    hKey.Destroy();

    Result := True;

  end;

  function TStoreRegistry.IsKeyExist(sKeyName: String): Boolean;
  var
    hKey: TRegistry;
  begin

    hKey := OpenKey(sKeyName);
    if(hKey = nil) Then
    begin
      Exit(False);
    end;

    hKey.Destroy();
    Result := True;

  end;


end.
