unit MainChatForm;

interface

uses

  Winapi.Windows,
  Winapi.Messages,
  System.SysUtils,
  System.Variants,
  System.Classes,
  Vcl.Graphics,
  Vcl.Controls,
  Vcl.Forms,
  Vcl.Dialogs,
  Vcl.StdCtrls,
  Vcl.ComCtrls,
  VaxSIPUserAgentExCOM, StoreChat,
  Vcl.ExtCtrls,
  StrUtils;

type

  TMainChatForm = class(TForm)

    GroupChatMsg: TGroupBox;
    LabelChatTyping: TLabel;
    ListBoxMessages: TListBox;
    EditSendMessage: TEdit;
    BtnSendMessage: TButton;
    GroupChatLog: TGroupBox;
    ListBoxStatusLog: TListBox;
    BtnClearLog: TButton;
    BtnClearMessage: TButton;
    TimerMsgTyping: TTimer;
    ListBoxContacts: TListView;
    EditContactName: TEdit;
    BtnAddContact: TButton;
    BtnRemoveContact: TButton;
    ComboMyChatStatus: TComboBox;

    constructor Create(objTabSheetChat: TTabSheet; objVaxCOM: TVaxSIPUserAgentExCOM);overload;

    procedure BtnAddContactClick(Sender: TObject);
    procedure BtnRemoveContactClick(Sender: TObject);

    procedure BtnSendMessageClick(Sender: TObject);
    procedure ComboMyChatStatusChange(Sender: TObject);

    procedure BtnClearLogClick(Sender: TObject);
    procedure BtnClearMessageClick(Sender: TObject);

    procedure TimerMsgTypingTimer(Sender: TObject);
    procedure EditSendMessageKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);

    procedure OnVaxOnline();
    procedure OnVaxOffline();

    procedure OnChatSubscribeSuccess(sUserName: String);
    procedure OnChatSubscribeFailed(sUserName: String; nStatusCode: Integer; sReasonPhrase: String);

    procedure OnChatRecvMsgText(sUserName: String; sMsgText: String);

    procedure OnChatRecvMsgTypingStart(sUserName: String);
    procedure OnChatRecvMsgTypingStop();

    procedure OnChatSendMsgTextFailed(sUserName: String; nStatusCode: Integer; sReasonPhrase: String; sMsgText: String);
    procedure OnChatSendMsgTextSuccess(sUserName: String; sMsgText: String);

    procedure OnChatContactStatus(sUserName: String; nStatusId: Integer);
    procedure OnChatSendMsgTypingSuccess(sUserName: String);
    procedure OnChatSendMsgTypingFailed(sUserName: String; nStatusCode: Integer; sReasonPhrase: String);

    procedure ListBoxContactsClick(Sender: TObject);

    private
    m_objVaxCOM: TVaxSIPUserAgentExCOM;
    m_bTyping: Boolean;

    private
    procedure LoadContactAll();
    procedure LoadContactMsgAll(sContactName: String);

    procedure UpdateStatusCombo();
    function GetStatusText(nStatusId: Integer): String;
    procedure AdjustTypingState();
    procedure AddToStatusLog(sChatLog: String);

  end;

implementation
 {$R *.dfm}

  constructor TMainChatForm.Create(objTabSheetChat: TTabSheet; objVaxCOM: TVaxSIPUserAgentExCOM);
  var
  Column: TListColumn;
  begin

    inherited Create(objTabSheetChat);
    m_objVaxCOM := objVaxCOM;

    Column := ListBoxContacts.Columns.Add();
    Column.Caption := 'UserName';
    Column.Width := 90;

    Column := ListBoxContacts.Columns.Add();
    Column.Caption := 'Status';
    Column.Width := 110;

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.OnVaxOnline();
  begin
    LabelChatTyping.Caption := '';
    m_bTyping := False;

    UpdateStatusCombo();
    LoadContactAll();
  end;

  procedure TMainChatForm.OnVaxOffline();
  begin
    ListBoxStatusLog.Items.Clear();
    ListBoxContacts.Items.Clear();
  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.LoadContactAll();
  var
    nIndex: Integer;
    objList: TStringList;
    Item: TListItem;
    sContactName: String;
  begin

    ListBoxContacts.Items.Clear();

    objList := TStringList.Create();
    TStoreChat.GetContactAll(objList);

    if(objList.Count <= 0) Then
        Exit();

    for nIndex := 0 to objList.Count - 1 do
    begin

      Item := ListBoxContacts.Items.Add;
      sContactName := objList[nIndex];

      Item.Caption := sContactName;
      Item.SubItems.Add('');

      m_objVaxCOM.ChatAddContact(sContactName);

    end;

    if (ListBoxContacts.ItemIndex = -1) and (ListBoxContacts.Items.Count <> 0) Then
    begin
      ListBoxContacts.ItemIndex := 0;
      LoadContactMsgAll(objList[0]);
    end;

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.UpdateStatusCombo();
  begin

    ComboMyChatStatus.Clear();

    ComboMyChatStatus.Items.Add('Online');
    ComboMyChatStatus.Items.Add('Offline');
    ComboMyChatStatus.Items.Add('Away');
    ComboMyChatStatus.Items.Add('On the phone');
    ComboMyChatStatus.Items.Add('Busy');

    ComboMyChatStatus.ItemIndex := TVaxSIPUserAgentExCOM.VAX_CONTACT_STATUS_OFFLINE;

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.BtnAddContactClick(Sender: TObject);
  var
    Item: TListItem;
  begin

    if EditContactName.Text = '' Then
      Exit;

    if (m_objVaxCOM.ChatAddContact(EditContactName.Text) = False) Then
    begin
      Exit();
    end;

    Item := ListBoxContacts.Items.Add;
    Item.Caption := EditContactName.Text;
    Item.SubItems.Add('');

    TStoreChat.AddContact(EditContactName.Text);

    if (ListBoxContacts.ItemIndex = -1) and (ListBoxContacts.Items.Count <> 0) Then
      ListBoxContacts.ItemIndex := 0;

    EditContactName.Text := '';

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.BtnRemoveContactClick(Sender: TObject);
  var
    sContactName: String;
  begin

    if ListBoxContacts.ItemIndex = -1 Then
      Exit;

    if MessageDlg('Do you really want to remove this Contact?', mtConfirmation, [mbYes, mbNo], 0, mbNo) = mrNo Then
      begin
        Exit();
      end;

    sContactName := ListBoxContacts.Items[ListBoxContacts.ItemIndex].Caption;

    if m_objVaxCOM.ChatRemoveContact(sContactName) = False Then
      Exit;

    ListBoxContacts.DeleteSelected;
    TStoreChat.RemoveContact(sContactName);

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.ListBoxContactsClick(Sender: TObject);
  var
    sContactName: String;
  begin

    if ListBoxContacts.ItemIndex = -1 Then
      Exit;

    ListBoxMessages.Items.Clear();

    sContactName := ListBoxContacts.Items[ListBoxContacts.ItemIndex].Caption;
    LoadContactMsgAll(sContactName);

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.LoadContactMsgAll(sContactName: String);
  var
    nIndex: Integer;
    objList: TStringList;
  begin

    ListBoxMessages.Items.Clear();

    objList := TStringList.Create();
    TStoreChat.GetMsgAll(sContactName, objList);

    if(objList.Count <= 0) Then
      Exit();

    for nIndex := 0 to objList.Count - 1 do
      ListBoxMessages.Items.Add(objList[nIndex]);

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.ComboMyChatStatusChange(Sender: TObject);
  begin

    if ComboMyChatStatus.ItemIndex = -1 Then
      Exit;

    m_objVaxCOM.ChatSetMyStatus(ComboMyChatStatus.ItemIndex);

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.BtnSendMessageClick(Sender: TObject);
  var
    sContactName: String;
    sMsg: String;
  begin

    if ListBoxContacts.ItemIndex = -1 Then
      Exit;

    if EditSendMessage.Text = '' Then
      Exit();

    sContactName := ListBoxContacts.Items[ListBoxContacts.ItemIndex].Caption;

    if m_objVaxCOM.ChatSendMessageText(sContactName, EditSendMessage.Text) = False Then
      Exit;

    sMsg := 'Me: ' + EditSendMessage.Text;

    ListBoxMessages.Items.Add(sMsg);
    TStoreChat.AddMsg(sContactName, sMsg);

    EditSendMessage.Text := '';

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.OnChatSendMsgTextSuccess(sUserName: String; sMsgText: String);
  begin
    AddToStatusLog('Send Message to ' + sUserName + ' Success.');
  end;

  procedure TMainChatForm.OnChatSendMsgTextFailed(sUserName: String; nStatusCode: Integer; sReasonPhrase: String; sMsgText: String);
  begin
    AddToStatusLog('Send Message to ' + sUserName + ' Failed.');
  end;

  procedure TMainChatForm.AddToStatusLog(sChatLog: String);
  begin
    ListBoxStatusLog.Items.Add(sChatLog);
    ListBoxStatusLog.TopIndex := ListBoxStatusLog.Items.Count - 1;
  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.BtnClearMessageClick(Sender: TObject);
  var
    sContactName: String;
  begin

    if ListBoxContacts.ItemIndex = -1 Then
      Exit;

    ListBoxMessages.Items.Clear();

    sContactName := ListBoxContacts.Items[ListBoxContacts.ItemIndex].Caption;
    TStoreChat.RemoveMsgAll(sContactName);

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.BtnClearLogClick(Sender: TObject);
  begin
    ListBoxStatusLog.Items.Clear();
  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  function TMainChatForm.GetStatusText(nStatusId: Integer): String;
  var
    sStatus: String;
  begin

    sStatus := '';

    Case nStatusId of
      TVaxSIPUserAgentExCOM.VAX_CONTACT_STATUS_ONLINE: sStatus := 'Online';
      TVaxSIPUserAgentExCOM.VAX_CONTACT_STATUS_OFFLINE: sStatus := 'Offline';
      TVaxSIPUserAgentExCOM.VAX_CONTACT_STATUS_AWAY: sStatus := 'Away';
      TVaxSIPUserAgentExCOM.VAX_CONTACT_STATUS_ON_PHONE: sStatus := 'On the phone';
      TVaxSIPUserAgentExCOM.VAX_CONTACT_STATUS_BUSY: sStatus := 'Busy';
      TVaxSIPUserAgentExCOM.VAX_CONTACT_STATUS_UNKNOWN: sStatus := 'Offline';
    end;

    Result := sStatus;

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.OnChatSubscribeSuccess(sUserName: String);
  begin
    AddToStatusLog('Chat Subscribe ' + sUserName + ' Success.');
  end;

  procedure TMainChatForm.OnChatSubscribeFailed(sUserName: String; nStatusCode: Integer; sReasonPhrase: String);
  begin
    AddToStatusLog('Chat Subscribe ' + sUserName + ' Failed.');
  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.OnChatContactStatus(sUserName: String; nStatusId: Integer);
  var
    sContactName: String;
    nContactCount: Integer;
    Item: TListItem;
  begin

    for nContactCount := 0 to (ListBoxContacts.Items.Count - 1) do
    begin

      sContactName := UpperCase(ListBoxContacts.Items.Item[nContactCount].Caption);

      if UpperCase(sUserName) = sContactName Then
      begin
        Item := ListBoxContacts.items.Item[nContactCount];
        Item.SubItems.Strings[0] := ' [' + GetStatusText(nStatusId) + ']';
      end;

    end;

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.OnChatRecvMsgText(sUserName: String; sMsgText: String);
  var
    nStart: Integer;
    nEnd: Integer;
    sMessage: String;
  begin

    nStart := Pos('>', sMsgText);

    if nStart = 0 Then
      begin
        sMessage := sUserName + ': ' + sMsgText;
      end
    else
      begin
        // Remote softphone is sending HTML based text messages

        nStart := nStart + 1;
        nEnd := PosEx('<', sMsgText, nStart);
        sMessage := sUserName + ': ' + Copy(sMsgText, nStart, nEnd - nStart)
      end;

    ListBoxMessages.Items.Add(sMessage);
    ListBoxMessages.ItemIndex := ListBoxMessages.Items.Count - 1;

    TStoreChat.AddMsg(sUserName, sMessage);

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.OnChatRecvMsgTypingStart(sUserName: String);
  begin

    m_bTyping := True;
    LabelChatTyping.Caption := sUserName + ': Typing';

    TimerMsgTyping.Enabled := False;
    TimerMsgTyping.Enabled := True;

    AdjustTypingState();

  end;

  procedure TMainChatForm.TimerMsgTypingTimer(Sender: TObject);
  begin
    AdjustTypingState();
  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.AdjustTypingState();
  begin

    if m_bTyping = False Then
    begin
      TimerMsgTyping.Enabled := False;
      OnChatRecvMsgTypingStop();
    end
    Else
      m_bTyping := False

  end;

  procedure TMainChatForm.OnChatRecvMsgTypingStop();
  begin
    LabelChatTyping.Caption := '';
  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.EditSendMessageKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
  var
    sContactName: String;
  begin

    if ListBoxContacts.ItemIndex = -1 Then
      Exit;

    sContactName := ListBoxContacts.Items[ListBoxContacts.ItemIndex].Caption;
    m_objVaxCOM.ChatSendMessageTyping(sContactName);

  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

  procedure TMainChatForm.OnChatSendMsgTypingSuccess(sUserName: String);
  begin
    AddToStatusLog('Send Typing to ' + sUserName + ' Success.');
  end;

  procedure TMainChatForm.OnChatSendMsgTypingFailed(sUserName: String; nStatusCode: Integer; sReasonPhrase: String);
  begin
    AddToStatusLog('Send Typing to ' + sUserName + ' Failed.');
  end;

  //////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////////

end.
